import numpy as np
import matplotlib
import matplotlib.pyplot as plt

f = open('annotation.txt')
line = [i.strip() for i in f.readlines()]
f.close()

data = []

for i in line:
	data.append([int(j) for j in i.split('\t')])

data = np.array(data)


col_label = ['WT','1bp insertion','1bp deletion','2bp insertion','2bp deletion',
'3bp insertion','3bp deletion','4bp insertion','4bp deletion','5bp insertion','5bp deletion']
row_label = ['WT','1bp insertion','1bp deletion','2bp insertion','2bp deletion',
'3bp insertion','3bp deletion','4bp insertion','4bp deletion','5bp insertion','5bp deletion']

def heatmap(data, row_labels, col_labels, ax=None, cbar_kw={}, cbarlabel="", **kwargs):
	if not ax:
		ax = plt.gca()

	im = ax.imshow(data, **kwargs)
	cbar = ax.figure.colorbar(im, ax=ax, **cbar_kw)
	cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom")
	ax.tick_params(top=True, bottom=False, labeltop=True, labelbottom=False)
	plt.setp(ax.get_xticklabels(), rotation=-30, ha="right", rotation_mode="anchor")
	ax.spines['bottom'].set_visible(False)
	ax.spines['top'].set_visible(False)
	ax.spines['left'].set_visible(False)
	ax.spines['right'].set_visible(False)
	ax.set_xticks(np.arange(data.shape[1]+1)-.5, minor=True)
	ax.set_yticks(np.arange(data.shape[0]+1)-.5, minor=True)
	ax.grid(which="minor", color="w", linestyle='-', linewidth=2)
	ax.tick_params(which="minor", bottom=False, left=False)

	return im, cbar


def annotate_heatmap(im, data=None, valfmt="{x:.0f}", textcolors=("black", "white"),
                    threshold=None, **textkw):
	
	if not isinstance(data, (list, np.ndarray)):
		data = im.get_array()

	if threshold is not None:
		threshold = im.norm(threshold)
	else:
		threshold = im.norm(data.max())/2.

	kw = dict(horizontalalignment="center",
              verticalalignment="center")
	kw.update(textkw)
	
	if isinstance(valfmt, str):
		valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)
	
	texts = []
	for i in range(data.shape[0]):
		for j in range(data.shape[1]):
			kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
			text = im.axes.text(j, i, valfmt(data[i, j], None), fontsize=13, **kw)
			texts.append(text)

	return texts


fig, ax = plt.subplots()
im, cbar = heatmap(data, row_label, col_label, ax=ax, cmap='magma_r', 
	cbarlabel='number of allele')
texts = annotate_heatmap(im, valfmt="{x:.0f}")

plt.savefig('mutation_anno.pdf', format='pdf')
plt.show()
